<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Http\Request;
use App\Http\Response;
use App\Http\HttpException;
use App\Security\ApiAuth;
use App\Database\Connection;
use App\Services\HaciendaApiClient;
use PDO;
use RuntimeException;

class MhController
{
    /**
     * POST /api/v1/mh/test-credenciales
     *
     * Body JSON (opcional):
     * {
     *   "mh_usuario": "usuario-mh",
     *   "mh_password": "password-mh"
     * }
     *
     * - Si se envían, se guardan en clientes_api.
     * - Luego se prueba contra el sandbox de Hacienda pidiendo un token.
     */
    public function testCredenciales(Request $request): void
    {
        // 1. Validar API key y obtener client_id + ambiente
        ApiAuth::requireValidKey($request);

        $clienteId = $request->getClientId();
        $ambiente  = $request->getClientAmbiente() ?? 'pruebas';

        if ($clienteId === null) {
            throw new RuntimeException('Cliente no definido en Request');
        }

        // 2. Leer JSON (si viene)
        $contentType = $request->header('Content-Type', '');
        $isJson      = str_starts_with(strtolower($contentType), 'application/json');

        $body = $isJson ? ($request->json() ?? []) : [];

        $mhUsuario  = isset($body['mh_usuario'])  ? trim((string)$body['mh_usuario'])  : null;
        $mhPassword = isset($body['mh_password']) ? trim((string)$body['mh_password']) : null;

        $pdo = Connection::getPdo();

        // 3. Si vienen credenciales nuevas, guardarlas en clientes_api
        if ($mhUsuario !== null && $mhPassword !== null && $mhUsuario !== '' && $mhPassword !== '') {
            $sql = "
                UPDATE clientes_api
                SET mh_usuario = :u,
                    mh_password = :p
                WHERE id = :id
            ";

            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                ':u'  => $mhUsuario,
                ':p'  => $mhPassword,
                ':id' => $clienteId,
            ]);
        }

        // 4. Pedir token a Hacienda usando lo que haya en BD
        $tokenInfo = HaciendaApiClient::obtenerTokenParaCliente($clienteId, $ambiente);

        // 5. Responder sin exponer el token completo
        Response::json([
            'ok'           => true,
            'mensaje'      => 'Credenciales válidas en el sandbox de Hacienda',
            'ambiente'     => $ambiente,
            'token_inicio' => substr($tokenInfo['access_token'], 0, 25) . '...',
            'expira_en'    => $tokenInfo['expires_in'],
        ]);
    }
}
