<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Http\Request;
use App\Http\Response;
use App\Http\HttpException;
use App\Security\ApiAuth;
use App\Database\Connection;
use App\Services\HaciendaConsecutivo;
use App\Services\XmlFacturaBuilder;
use App\Services\FirmaService;
use App\Services\HaciendaApiClient;
use RuntimeException;
use DateTimeImmutable;
use PDO;

class DocumentController
{
    /**
     * POST index.php?r=api/v1/documentos
     * Crea documento, genera clave, consecutivo, XML y lo firma (XAdES).
     * Además guarda las líneas en documentos_detalle.
     */
    public function store(Request $request): void
    {
        // 1. Autenticación por API key
        ApiAuth::requireValidKey($request);

        // 2. Content-Type
        $contentType = $request->header('Content-Type', '');
        if (!str_starts_with(strtolower($contentType), 'application/json')) {
            throw new HttpException('Content-Type debe ser application/json', 415);
        }

        // 3. JSON
        $data = $request->json();
        if ($data === null) {
            throw new HttpException('JSON inválido', 400);
        }

        // 4. Validaciones mínimas
        $required = ['tipo', 'consecutivo_interno', 'emisor', 'receptor', 'lineas'];
        foreach ($required as $field) {
            if (!array_key_exists($field, $data)) {
                throw new HttpException("Falta campo requerido: {$field}", 422);
            }
        }

        if (!is_array($data['emisor']) || !is_array($data['receptor'])) {
            throw new HttpException('Emisor y receptor deben ser objetos', 422);
        }

        if (!is_array($data['lineas']) || count($data['lineas']) === 0) {
            throw new HttpException('Debe haber al menos una línea', 422);
        }

        // 5. Info del cliente desde la API key
        $clienteId = $request->getClientId();
        if ($clienteId === null) {
            throw new RuntimeException('Cliente no definido en Request (error interno de autenticación)');
        }

        // 6. Cargar configuración real del cliente desde clientes_api
        $pdo = Connection::getPdo();

        $sqlCli = "
            SELECT
                ambiente,
                sucursal,
                terminal
                -- si tienes columnas de identificación, puedes agregarlas aquí, por ejemplo:
                -- , identificacion
                -- , tipo_identificacion
            FROM clientes_api
            WHERE id = :id
              AND activo = 1
            LIMIT 1
        ";
        $stmtCli = $pdo->prepare($sqlCli);
        $stmtCli->execute([':id' => $clienteId]);
        $cliCfg = $stmtCli->fetch(PDO::FETCH_ASSOC);

        if (!$cliCfg) {
            throw new RuntimeException('No se encontró configuración activa del cliente en clientes_api');
        }

        $ambiente = $cliCfg['ambiente'] ?? 'pruebas';

        // 6. Datos base
        $tipo               = (string) $data['tipo'];
        $consecutivoInterno = (string) $data['consecutivo_interno'];
        $estado             = 'firmado'; // ya con firma XAdES real

        $jsonPayload = json_encode($data, JSON_UNESCAPED_UNICODE);
        if ($jsonPayload === false) {
            throw new RuntimeException('No se pudo codificar el JSON del documento');
        }

        $pdo->beginTransaction();

        try {
            // 7. Secuencia para este cliente + tipo + ambiente
            $sqlSeq = "
                SELECT MAX(secuencia) AS max_seq
                FROM documentos
                WHERE cliente_id = :cliente_id
                  AND tipo = :tipo
                  AND ambiente = :ambiente
            ";

            $stmtSeq = $pdo->prepare($sqlSeq);
            $stmtSeq->execute([
                ':cliente_id' => $clienteId,
                ':tipo'       => $tipo,
                ':ambiente'   => $ambiente,
            ]);

            $rowSeq    = $stmtSeq->fetch(PDO::FETCH_ASSOC);
            $maxSeq    = ($rowSeq && $rowSeq['max_seq'] !== null) ? (int) $rowSeq['max_seq'] : 0;
            $secuencia = $maxSeq + 1;

            // 8. Sucursal / terminal desde clientes_api
            $sucursal = (int) ($cliCfg['sucursal'] ?? 1);
            $terminal = (int) ($cliCfg['terminal'] ?? 1);

            // 9. Identificación del emisor (CRÍTICO para que coincida con la clave)
            $identEmisor = '';

            // 9.1 Preferimos el campo plano numero_identificacion (como envías en el test)
            if (!empty($data['emisor']['numero_identificacion'])) {
                $identEmisor = (string) $data['emisor']['numero_identificacion'];
            }
            // 9.2 Si no viene, intentamos con emisor.identificacion.* (estructura clásica)
            elseif (isset($data['emisor']['identificacion'])) {
                $ident = $data['emisor']['identificacion'];

                if (is_array($ident)) {
                    if (!empty($ident['numero'])) {
                        $identEmisor = (string) $ident['numero'];
                    } elseif (!empty($ident['Numero'])) { // por si viene con N mayúscula
                        $identEmisor = (string) $ident['Numero'];
                    }
                } elseif ($ident !== '') {
                    // por si alguien envía simplemente "identificacion" = "206530483"
                    $identEmisor = (string) $ident;
                }
            }

            // 9.3 Como fallback, podemos intentar usar la identificación almacenada en clientes_api, si existe
            if ($identEmisor === '' && isset($cliCfg['identificacion']) && $cliCfg['identificacion'] !== '') {
                $identEmisor = (string) $cliCfg['identificacion'];
            }

            // 9.4 Si aún está vacío, NO generamos clave con basura: error explícito
            if ($identEmisor === '') {
                throw new RuntimeException(
                    'No se pudo determinar la identificación del emisor para generar la clave. ' .
                    'Verifica que emisor.numero_identificacion o emisor.identificacion.numero vengan en el payload.'
                );
            }

            // 10. Fecha emisión
            $fechaEmision = new DateTimeImmutable('now', new \DateTimeZone('America/Costa_Rica'));

            // 11. Consecutivo y clave (usando tus services)
            $numeroConsecutivo = HaciendaConsecutivo::generarNumeroConsecutivo(
                $sucursal,
                $terminal,
                $tipo,
                $secuencia
            );

            $clave = HaciendaConsecutivo::generarClave(
                $fechaEmision,
                $identEmisor,
                $numeroConsecutivo,
                '1' // situación normal
            );

            // 12. XML sin firma
            $xmlSinFirma = XmlFacturaBuilder::construirFacturaXml(
                $data,
                $clave,
                $numeroConsecutivo,
                $ambiente
            );

            // 13. Firmar XML (XAdES real)
            $xmlFirmado = FirmaService::firmarXmlParaCliente(
                $xmlSinFirma,
                (int) $clienteId
            );

            // 14. Insertar cabecera en documentos
            $sql = "
                INSERT INTO documentos
                  (cliente_id,
                   secuencia,
                   tipo,
                   consecutivo_interno,
                   numero_consecutivo,
                   clave,
                   estado,
                   hacienda_estado,
                   json_payload,
                   xml_sin_firma,
                   xml_firmado,
                   ambiente,
                   creado_en)
                VALUES
                  (:cliente_id,
                   :secuencia,
                   :tipo,
                   :consecutivo_interno,
                   :numero_consecutivo,
                   :clave,
                   :estado,
                   NULL,
                   :json_payload,
                   :xml_sin_firma,
                   :xml_firmado,
                   :ambiente,
                   NOW())
            ";

            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                ':cliente_id'          => $clienteId,
                ':secuencia'           => $secuencia,
                ':tipo'                => $tipo,
                ':consecutivo_interno' => $consecutivoInterno,
                ':numero_consecutivo'  => $numeroConsecutivo,
                ':clave'               => $clave,
                ':estado'              => $estado,
                ':json_payload'        => $jsonPayload,
                ':xml_sin_firma'       => $xmlSinFirma,
                ':xml_firmado'         => $xmlFirmado,
                ':ambiente'            => $ambiente,
            ]);

            $documentId = (int) $pdo->lastInsertId();

            // 15. Guardar detalle en documentos_detalle
            $totalComprobante = 0.0;
            $lineas           = is_array($data['lineas']) ? $data['lineas'] : [];

            if (!empty($lineas)) {
                $sqlDet = "
                    INSERT INTO documentos_detalle
                      (documento_id, linea, codigo_cabys, descripcion,
                       cantidad, unidad, precio_unitario,
                       descuento_monto, impuesto_monto, total_linea, creado_en)
                    VALUES
                      (:documento_id, :linea, :codigo_cabys, :descripcion,
                       :cantidad, :unidad, :precio_unitario,
                       :descuento_monto, :impuesto_monto, :total_linea, NOW())
                ";

                $stmtDet = $pdo->prepare($sqlDet);
                $lineaNum = 1;

                foreach ($lineas as $line) {
                    if (!is_array($line)) {
                        continue;
                    }

                    // Mapeo defensivo del CABYS
                    $codigoCabys = null;
                    if (isset($line['codigo_cabys'])) {
                        $codigoCabys = (string) $line['codigo_cabys'];
                    } elseif (isset($line['codigo'])) {
                        $codigoCabys = (string) $line['codigo'];
                    }

                    $descripcion = (string) (
                        $line['descripcion']
                        ?? $line['detalle']
                        ?? ''
                    );

                    $cantidad = isset($line['cantidad']) ? (float) $line['cantidad'] : 0.0;

                    $unidad = (string) (
                        $line['unidad_medida']
                        ?? $line['unidad']
                        ?? 'Sp'
                    );

                    $precioUnit = (float) (
                        $line['precio_unitario']
                        ?? $line['precio']
                        ?? 0.0
                    );

                    $descuento = (float) (
                        $line['descuento_monto']
                        ?? $line['monto_descuento']
                        ?? 0.0
                    );

                    $impuestoMonto = (float) (
                        $line['impuesto_monto']
                        ?? $line['monto_impuesto']
                        ?? 0.0
                    );

                    if ($descripcion === '' || $cantidad <= 0) {
                        continue;
                    }

                    $subtotal   = $cantidad * $precioUnit;
                    $totalLinea = max(0.0, $subtotal - $descuento + $impuestoMonto);
                    $totalComprobante += $totalLinea;

                    $stmtDet->execute([
                        ':documento_id'    => $documentId,
                        ':linea'           => $lineaNum,
                        ':codigo_cabys'    => $codigoCabys,
                        ':descripcion'     => $descripcion,
                        ':cantidad'        => $cantidad,
                        ':unidad'          => $unidad,
                        ':precio_unitario' => $precioUnit,
                        ':descuento_monto' => $descuento,
                        ':impuesto_monto'  => $impuestoMonto,
                        ':total_linea'     => $totalLinea,
                    ]);

                    $lineaNum++;
                }
            }

            $pdo->commit();

            Response::json([
                'ok'                 => true,
                'documento_id'       => $documentId,
                'cliente_id'         => $clienteId,
                'ambiente'           => $ambiente,
                'estado'             => $estado,
                'secuencia'          => $secuencia,
                'numero_consecutivo' => $numeroConsecutivo,
                'clave'              => $clave,
                'total_comprobante'  => round($totalComprobante, 5),
                'mensaje'            => 'Documento almacenado con detalle y XML firmado correctamente',
            ], 201);
        } catch (HttpException $e) {
            $pdo->rollBack();
            throw $e;
        } catch (\Throwable $e) {
            $pdo->rollBack();

            $msg = sprintf(
                'Error interno al crear/firmar documento: %s (origen: %s:%d)',
                $e->getMessage(),
                $e->getFile(),
                $e->getLine()
            );

            throw new RuntimeException($msg, (int) $e->getCode(), $e);
        }
    }

    /**
     * POST index.php?r=api/v1/documentos/enviar
     * Body JSON: { "documento_id": 6 }
     *
     * Envía el XML firmado al sandbox de Hacienda usando HaciendaApiClient::enviarXmlFirmadoReal().
     */
    public function send(Request $request): void
    {
        ApiAuth::requireValidKey($request);

        $contentType = $request->header('Content-Type', '');
        if (!str_starts_with(strtolower($contentType), 'application/json')) {
            throw new HttpException('Content-Type debe ser application/json', 415);
        }

        $data = $request->json();
        if ($data === null) {
            throw new HttpException('JSON inválido', 400);
        }

        if (!isset($data['documento_id'])) {
            throw new HttpException('Falta documento_id', 422);
        }

        $documentoId = (int) $data['documento_id'];
        $clienteId   = $request->getClientId();

        if ($clienteId === null) {
            throw new RuntimeException('Cliente no definido en Request');
        }

        $pdo = Connection::getPdo();

        // 1) Buscar documento del cliente ACTUAL (multi-tenant correcto)
        $sql = "
            SELECT
              id,
              cliente_id,
              estado,
              ambiente,
              clave,
              xml_firmado
            FROM documentos
            WHERE id = :id AND cliente_id = :cliente_id
            LIMIT 1
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id'         => $documentoId,
            ':cliente_id' => $clienteId,
        ]);

        $doc = $stmt->fetch(PDO::FETCH_ASSOC);

        // 2) Si no aparece, revisamos si existe el ID pero con OTRO cliente (debug)
        if (!$doc) {
            $sqlAny = "
                SELECT id, cliente_id, estado, ambiente
                FROM documentos
                WHERE id = :id
                LIMIT 1
            ";
            $stmtAny = $pdo->prepare($sqlAny);
            $stmtAny->execute([':id' => $documentoId]);
            $docAny = $stmtAny->fetch(PDO::FETCH_ASSOC);

            if ($docAny) {
                throw new HttpException(
                    'Documento no pertenece a este cliente. '
                    . 'documento.cliente_id=' . $docAny['cliente_id']
                    . ' / cliente_actual=' . $clienteId,
                    403
                );
            }

            throw new HttpException('Documento no encontrado para este cliente', 404);
        }

        if (!in_array($doc['estado'], ['firmado', 'error'], true)) {
            throw new HttpException('Solo se pueden enviar documentos firmados o en error', 409);
        }

        if ($doc['xml_firmado'] === null || $doc['xml_firmado'] === '') {
            throw new HttpException('Documento sin XML firmado', 500);
        }

        $clave      = (string) $doc['clave'];
        $xmlFirmado = (string) $doc['xml_firmado'];
        $ambiente   = (string) $doc['ambiente'];

        // Extraer datos básicos desde el XML firmado (fecha, emisor, receptor)
        $parsed = $this->extraerDatosDesdeXmlFirmado($xmlFirmado);

        $fechaEmisionIso = $parsed['fecha_emision'] ?? date('c');
        $tipoIdEmisor    = $parsed['emisor_tipo']   ?? '01';
        $numIdEmisor     = $parsed['emisor_num']    ?? '';
        $tipoIdReceptor  = $parsed['receptor_tipo'] ?? null;
        $numIdReceptor   = $parsed['receptor_num']  ?? null;

        // Enviar REAL al sandbox de Hacienda
        $resultadoEnvio = HaciendaApiClient::enviarXmlFirmadoReal(
            $clienteId,
            $clave,
            $fechaEmisionIso,
            $xmlFirmado,
            $tipoIdEmisor,
            $numIdEmisor,
            $tipoIdReceptor,
            $numIdReceptor
        );

        $httpStatus   = $resultadoEnvio['http_status'];
        $location     = $resultadoEnvio['location'];
        $xErrorCause  = $resultadoEnvio['x_error_cause'];
        $body         = $resultadoEnvio['body'];

        // Determinar estado interno según la respuesta
        $haciendaEstado  = null;
        $haciendaMensaje = null;

        if ($httpStatus >= 200 && $httpStatus < 300) {
            $haciendaEstado  = 'enviado';
            $haciendaMensaje = 'Documento enviado a Hacienda correctamente (sandbox)';
        } else {
            $haciendaEstado  = 'error';
            if (is_string($body) && $body !== '') {
                $haciendaMensaje = mb_substr($body, 0, 250);
            } elseif (is_array($body) && isset($body['message'])) {
                $haciendaMensaje = mb_substr((string) $body['message'], 0, 250);
            } elseif ($xErrorCause) {
                $haciendaMensaje = mb_substr($xErrorCause, 0, 250);
            } else {
                $haciendaMensaje = 'Error al enviar a Hacienda (HTTP ' . $httpStatus . ')';
            }
        }

        // Guardar el cuerpo de respuesta como texto (puede ser JSON o string plano)
        $responseXml = null;
        if (is_string($body)) {
            $responseXml = $body;
        } elseif (is_array($body)) {
            $tmp = json_encode($body, JSON_UNESCAPED_UNICODE);
            if ($tmp !== false) {
                $responseXml = $tmp;
            }
        }

        // Actualizar documento
        $sqlUpdate = "
            UPDATE documentos
            SET estado                = :estado,
                hacienda_estado       = :hacienda_estado,
                hacienda_mensaje      = :hacienda_mensaje,
                hacienda_response_xml = :hacienda_response_xml,
                hacienda_fecha_envio  = NOW(),
                actualizado_en        = NOW()
            WHERE id = :id
        ";

        $estadoNuevo = ($haciendaEstado === 'enviado') ? 'enviado' : 'error';

        $stmtUp = $pdo->prepare($sqlUpdate);
        $stmtUp->execute([
            ':estado'               => $estadoNuevo,
            ':hacienda_estado'      => $haciendaEstado,
            ':hacienda_mensaje'     => $haciendaMensaje,
            ':hacienda_response_xml'=> $responseXml,
            ':id'                   => $documentoId,
        ]);

        Response::json([
            'ok'                => ($haciendaEstado === 'enviado'),
            'documento_id'      => $documentoId,
            'cliente_id'        => $clienteId,
            'ambiente'          => $ambiente,
            'estado'            => $estadoNuevo,
            'http_status'       => $httpStatus,
            'hacienda_estado'   => $haciendaEstado,
            'hacienda_mensaje'  => $haciendaMensaje,
            'location'          => $location,
            'x_error_cause'     => $xErrorCause,
            'body'              => $body,
            'mensaje'           => 'Documento enviado a Hacienda (sandbox) y estado actualizado',
        ]);
    }

    /**
     * GET index.php?r=api/v1/documentos/ver&id=14
     *
     * Devuelve cabecera + detalle del documento para el cliente autenticado.
     */
    public function show(Request $request): void
    {
        ApiAuth::requireValidKey($request);

        $clienteId = $request->getClientId();
        if ($clienteId === null) {
            throw new RuntimeException('Cliente no definido en Request');
        }

        $documentoId = isset($_GET['id']) ? (int) $_GET['id'] : 0;
        if ($documentoId <= 0) {
            throw new HttpException('Parámetro id inválido o faltante', 422);
        }

        $pdo = Connection::getPdo();

        // 1) Traer cabecera
        $sqlCab = "
            SELECT
              id,
              cliente_id,
              ambiente,
              estado,
              tipo,
              secuencia,
              consecutivo_interno,
              numero_consecutivo,
              clave,
              hacienda_estado,
              hacienda_mensaje,
              json_payload,
              hacienda_fecha_envio,
              hacienda_fecha_respuesta,
              creado_en
            FROM documentos
            WHERE id = :id AND cliente_id = :cliente_id
            LIMIT 1
        ";

        $stmtCab = $pdo->prepare($sqlCab);
        $stmtCab->execute([
            ':id'         => $documentoId,
            ':cliente_id' => $clienteId,
        ]);

        $cab = $stmtCab->fetch(PDO::FETCH_ASSOC);

        if (!$cab) {
            throw new HttpException('Documento no encontrado para este cliente', 404);
        }

        // Decodificar el JSON original (por si lo quieres ver ya estructurado)
        $payload = null;
        if (!empty($cab['json_payload'])) {
            $dec = json_decode($cab['json_payload'], true);
            if (is_array($dec)) {
                $payload = $dec;
            }
        }

        // 2) Traer detalle
        $sqlDet = "
            SELECT
              id,
              linea,
              codigo_cabys,
              descripcion,
              cantidad,
              unidad,
              precio_unitario,
              descuento_monto,
              impuesto_monto,
              total_linea,
              creado_en
            FROM documentos_detalle
            WHERE documento_id = :documento_id
            ORDER BY linea ASC, id ASC
        ";

        $stmtDet = $pdo->prepare($sqlDet);
        $stmtDet->execute([':documento_id' => $documentoId]);
        $detalle = $stmtDet->fetchAll(PDO::FETCH_ASSOC);

        Response::json([
            'ok'   => true,
            'data' => [
                'cabecera' => [
                    'id'                   => (int) $cab['id'],
                    'cliente_id'           => (int) $cab['cliente_id'],
                    'ambiente'             => (string) $cab['ambiente'],
                    'estado'               => (string) $cab['estado'],
                    'tipo'                 => (string) $cab['tipo'],
                    'secuencia'            => (int) $cab['secuencia'],
                    'consecutivo_interno'  => (string) $cab['consecutivo_interno'],
                    'numero_consecutivo'   => (string) $cab['numero_consecutivo'],
                    'clave'                => (string) $cab['clave'],
                    'hacienda_estado'      => $cab['hacienda_estado'],
                    'hacienda_mensaje'     => $cab['hacienda_mensaje'],
                    'hacienda_fecha_envio' => $cab['hacienda_fecha_envio'],
                    'hacienda_fecha_resp'  => $cab['hacienda_fecha_respuesta'],
                    'creado_en'            => $cab['creado_en'],
                ],
                'payload_original' => $payload,
                'detalle'          => $detalle,
            ],
        ]);
    }

    /**
     * GET index.php?r=api/v1/documentos/estado&id=6
     *
     * Si el documento está en estado "enviado" (o similar) intenta consultar
     * el estado real en Hacienda usando la clave.
     */
    public function status(Request $request): void
    {
        ApiAuth::requireValidKey($request);

        $clienteId = $request->getClientId();
        if ($clienteId === null) {
            throw new RuntimeException('Cliente no definido en Request');
        }

        $documentoId = isset($_GET['id']) ? (int) $_GET['id'] : 0;
        if ($documentoId <= 0) {
            throw new HttpException('Parámetro id inválido o faltante', 422);
        }

        $pdo = Connection::getPdo();

        $sql = "
            SELECT
              id,
              cliente_id,
              ambiente,
              estado,
              hacienda_estado,
              hacienda_mensaje,
              clave,
              numero_consecutivo,
              hacienda_fecha_envio,
              hacienda_fecha_respuesta,
              hacienda_response_xml
            FROM documentos
            WHERE id = :id AND cliente_id = :cliente_id
            LIMIT 1
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id'         => $documentoId,
            ':cliente_id' => $clienteId,
        ]);

        $doc = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$doc) {
            throw new HttpException('Documento no encontrado para este cliente', 404);
        }

        $haciendaConsulta = null;
        $consultaError    = null;

        // Si está enviado o procesando y aún no hay fecha de respuesta, consultamos en Hacienda
        $estadoActual       = (string) $doc['estado'];
        $haciendaEstadoPrev = $doc['hacienda_estado'];

        $deberiaConsultar = in_array($estadoActual, ['enviado', 'firmado'], true)
            || in_array($haciendaEstadoPrev, [null, '', 'enviado', 'procesando', 'recibido'], true);

        if ($deberiaConsultar) {
            try {
                $clave    = (string) $doc['clave'];
                $ambiente = (string) $doc['ambiente'];

                $haciendaConsulta = HaciendaApiClient::consultarEstadoPorClave(
                    (int) $doc['cliente_id'],
                    $clave,
                    $ambiente ?: 'pruebas'
                );

                $httpStatus  = $haciendaConsulta['http_status'];
                $body        = $haciendaConsulta['body'];
                $xErrorCause = $haciendaConsulta['x_error_cause'] ?? null;

                if ($httpStatus >= 200 && $httpStatus < 300 && is_array($body)) {
                    $indEstado    = isset($body['ind-estado']) ? (string) $body['ind-estado'] : null;
                    $fechaRespStr = isset($body['fecha']) ? (string) $body['fecha'] : null;

                    $nuevoEstadoDoc      = $estadoActual;
                    $nuevoHaciendaEstado = $indEstado ?: $haciendaEstadoPrev;
                    $nuevoMensaje        = $doc['hacienda_mensaje'];

                    if ($indEstado !== null) {
                        $nuevoMensaje = 'Estado en Hacienda: ' . $indEstado;
                    } elseif ($xErrorCause) {
                        $nuevoMensaje = $xErrorCause;
                    }

                    // Mapear a estados internos
                    if ($indEstado === 'aceptado') {
                        $nuevoEstadoDoc = 'aceptado';
                    } elseif ($indEstado === 'rechazado') {
                        $nuevoEstadoDoc = 'rechazado';
                    } elseif (in_array($indEstado, ['procesando', 'recibido'], true)) {
                        if ($estadoActual === 'firmado') {
                            $nuevoEstadoDoc = 'enviado';
                        }
                    } elseif ($indEstado === 'error') {
                        $nuevoEstadoDoc = 'error';
                    }

                    // Decodificar respuesta-xml si viene
                    $respuestaXmlDec = null;
                    if (isset($body['respuesta-xml']) && is_string($body['respuesta-xml']) && $body['respuesta-xml'] !== '') {
                        $tmpXml = base64_decode($body['respuesta-xml'], true);
                        if ($tmpXml !== false) {
                            $respuestaXmlDec = $tmpXml;
                        }
                    }

                    $fechaRespuesta = $doc['hacienda_fecha_respuesta'];
                    if ($fechaRespStr && $fechaRespStr !== '') {
                        $fechaRespuesta = $fechaRespStr;
                    }

                    $sqlUpdate = "
                        UPDATE documentos
                        SET estado                   = :estado,
                            hacienda_estado          = :hacienda_estado,
                            hacienda_mensaje         = :hacienda_mensaje,
                            hacienda_response_xml    = :hacienda_response_xml,
                            hacienda_fecha_respuesta = :hacienda_fecha_respuesta,
                            actualizado_en           = NOW()
                        WHERE id = :id
                    ";

                    $stmtUp = $pdo->prepare($sqlUpdate);
                    $stmtUp->execute([
                        ':estado'                   => $nuevoEstadoDoc,
                        ':hacienda_estado'          => $nuevoHaciendaEstado,
                        ':hacienda_mensaje'         => $nuevoMensaje,
                        ':hacienda_response_xml'    => $respuestaXmlDec ?? $doc['hacienda_response_xml'],
                        ':hacienda_fecha_respuesta' => $fechaRespuesta,
                        ':id'                       => $documentoId,
                    ]);

                    $doc['estado']                   = $nuevoEstadoDoc;
                    $doc['hacienda_estado']          = $nuevoHaciendaEstado;
                    $doc['hacienda_mensaje']         = $nuevoMensaje;
                    $doc['hacienda_response_xml']    = $respuestaXmlDec ?? $doc['hacienda_response_xml'];
                    $doc['hacienda_fecha_respuesta'] = $fechaRespuesta;
                } else {
                    $consultaError = 'HTTP ' . $httpStatus . ' al consultar Hacienda';
                    if (is_array($body) && isset($body['message'])) {
                        $consultaError .= ': ' . $body['message'];
                    } elseif (is_string($body) && $body !== '') {
                        $consultaError .= ': ' . $body;
                    } elseif ($xErrorCause) {
                        $consultaError .= ' (' . $xErrorCause . ')';
                    }
                }
            } catch (\Throwable $e) {
                $consultaError = $e->getMessage();
            }
        }

        $response = [
            'ok'                   => true,
            'documento_id'         => (int) $doc['id'],
            'cliente_id'           => (int) $doc['cliente_id'],
            'ambiente'             => (string) $doc['ambiente'],
            'estado'               => (string) $doc['estado'],
            'hacienda_estado'      => $doc['hacienda_estado'],
            'hacienda_mensaje'     => $doc['hacienda_mensaje'],
            'clave'                => $doc['clave'],
            'numero_consecutivo'   => $doc['numero_consecutivo'],
            'hacienda_fecha_envio' => $doc['hacienda_fecha_envio'],
            'hacienda_fecha_resp'  => $doc['hacienda_fecha_respuesta'],
        ];

        if ($haciendaConsulta !== null) {
            $response['consulta_http_status'] = $haciendaConsulta['http_status'];
            $response['consulta_body']        = $haciendaConsulta['body'];
            $response['consulta_x_error']     = $haciendaConsulta['x_error_cause'] ?? null;
        }

        if ($consultaError !== null) {
            $response['consulta_error'] = $consultaError;
        }

        Response::json($response);
    }

    /**
     * Extrae datos básicos desde el XML firmado de Hacienda.
     *
     * @return array{
     *   fecha_emision?:string,
     *   emisor_tipo?:string,
     *   emisor_num?:string,
     *   receptor_tipo?:string,
     *   receptor_num?:string
     * }
     */
    private function extraerDatosDesdeXmlFirmado(string $xml): array
    {
        $data = [];

        // FechaEmision
        if (preg_match('~<FechaEmision>([^<]+)</FechaEmision>~', $xml, $m)) {
            $data['fecha_emision'] = trim($m[1]);
        }

        // Emisor -> Identificacion
        if (preg_match('~<Emisor>.*?<Identificacion>.*?<Tipo>([^<]+)</Tipo>.*?<Numero>([^<]*)</Numero>.*?</Identificacion>.*?</Emisor>~s', $xml, $m)) {
            $data['emisor_tipo'] = trim($m[1]);
            $data['emisor_num']  = trim($m[2]);
        }

        // Receptor -> Identificacion (opcional)
        if (preg_match('~<Receptor>.*?<Identificacion>.*?<Tipo>([^<]+)</Tipo>.*?<Numero>([^<]*)</Numero>.*?</Identificacion>.*?</Receptor>~s', $xml, $m)) {
            $data['receptor_tipo'] = trim($m[1]);
            $data['receptor_num']  = trim($m[2]);
        }

        return $data;
    }
}
