<?php
declare(strict_types=1);

namespace App\Services;

use App\Database\Connection;
use PDO;
use RuntimeException;

class ExoneracionService
{
    /**
     * Consulta una exoneración por código de autorización (AL-XXXXXXXX-XX)
     *
     * @return array<string,mixed>|null
     */
    public static function consultarPorAutorizacion(string $autorizacion, bool $forceRefresh = false): ?array
    {
        $autorizacion = strtoupper(trim($autorizacion));

        // Validar formato básico AL-XXXXXXXX-XX según docs oficiales
        if (!preg_match('/^AL-\d{8}-\d{2}$/', $autorizacion)) {
            return null;
        }

        $pdo = Connection::getPdo();

        // 1) Revisa caché local
        if (!$forceRefresh) {
            $sql = "
                SELECT *
                FROM exoneraciones_fe
                WHERE autorizacion = :aut
                LIMIT 1
            ";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([':aut' => $autorizacion]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($row) {
                // Si fue 200 y está reciente (ej: 30 días), devolvemos caché
                if ((int)$row['ultimo_http_code'] === 200 && self::esReciente($row['actualizado_en'])) {
                    return self::mapearDesdeFila($row);
                }
            }
        }

        // 2) Llamar al API de Hacienda /fe/ex
        $url = 'https://api.hacienda.go.cr/fe/ex?autorizacion=' . urlencode($autorizacion);
        $data = self::httpGetJson($url, $httpCode);

        // 3) Manejo de códigos
        if ($httpCode === 400) {
            self::guardarResultado($autorizacion, null, null, null, null, null, null, null, $data, 400, 'Autorización inválida (400)');
            return null;
        }

        if ($httpCode === 404) {
            self::guardarResultado($autorizacion, null, null, null, null, null, null, null, $data, 404, 'Exoneración no encontrada (404)');
            return null;
        }

        if ($httpCode === 429) {
            // Si tenemos caché anterior, lo devolvemos con warning
            $sql = "
                SELECT *
                FROM exoneraciones_fe
                WHERE autorizacion = :aut
                LIMIT 1
            ";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([':aut' => $autorizacion]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($row) {
                return self::mapearDesdeFila($row) + [
                    'warning' => 'Se devolvió caché porque Hacienda respondió 429 (límite de consumo).',
                ];
            }

            self::guardarResultado($autorizacion, null, null, null, null, null, null, null, $data, 429, 'Límite de consumo (429)');
            return null;
        }

        if ($httpCode !== 200 || !is_array($data)) {
            self::guardarResultado($autorizacion, null, null, null, null, null, null, null, $data, $httpCode, 'Error HTTP ' . $httpCode);
            return null;
        }

        // 4) Mapear respuesta de Hacienda
        $mapeado = self::mapearDesdeHacienda($autorizacion, $data);

        // 5) Guardar / actualizar
        self::guardarResultado(
            $autorizacion,
            $mapeado['numero_documento'],
            $mapeado['tipo_documento'],
            $mapeado['nombre_institucion'],
            $mapeado['porcentaje_exoneracion'],
            $mapeado['fecha_inicio'],
            $mapeado['fecha_fin'],
            $mapeado['cabys_autorizados'],
            $data,
            200,
            null,
            $mapeado['estado'],
            $mapeado['condicion']
        );

        return $mapeado + [
            'raw' => $data,
        ];
    }

    private static function esReciente(?string $fecha): bool
    {
        if ($fecha === null) {
            return false;
        }
        $ts = strtotime($fecha);
        if ($ts === false) {
            return false;
        }
        $hace30 = strtotime('-30 days');
        return $ts >= $hace30;
    }

    /**
     * @param array<string,mixed> $row
     * @return array<string,mixed>
     */
    private static function mapearDesdeFila(array $row): array
    {
        return [
            'autorizacion'           => $row['autorizacion'],
            'numero_documento'       => $row['numero_documento'],
            'tipo_documento'         => $row['tipo_documento'],
            'nombre_institucion'     => $row['nombre_institucion'],
            'porcentaje_exoneracion' => $row['porcentaje_exoneracion'] !== null ? (float)$row['porcentaje_exoneracion'] : null,
            'fecha_inicio'           => $row['fecha_inicio'],
            'fecha_fin'              => $row['fecha_fin'],
            'estado'                 => $row['estado'],
            'condicion'              => $row['condicion'],
            'cabys_autorizados'      => $row['cabys_autorizados']
                ? json_decode((string)$row['cabys_autorizados'], true)
                : null,
        ];
    }

    /**
     * La estructura exacta de /fe/ex puede variar; aquí hacemos un mapeo defensivo.
     *
     * @param array<string,mixed> $data
     * @return array<string,mixed>
     */
private static function mapearDesdeHacienda(string $autorizacion, array $data): array
{
    // Ojo: algunos campos pueden venir como string o como array
    $numeroDocumentoRaw = $data['numeroDocumento'] ?? ($data['numeroAutorizacion'] ?? null);
    $tipoDocRaw         = $data['tipoDocumento'] ?? ($data['tipo'] ?? null);
    $nombreInstRaw      = $data['nombreInstitucion'] ?? ($data['institucion'] ?? null);

    // Normalizar número de documento
    $numeroDocumento = null;
    if (is_array($numeroDocumentoRaw)) {
        $numeroDocumento = $numeroDocumentoRaw['codigo'] ?? ($numeroDocumentoRaw['valor'] ?? json_encode($numeroDocumentoRaw, JSON_UNESCAPED_UNICODE));
    } elseif ($numeroDocumentoRaw !== null) {
        $numeroDocumento = (string)$numeroDocumentoRaw;
    }

    // 
    $tipoDocumento = null;
    if (is_array($tipoDocRaw)) {
        // Intentamos tomar algo representativo; ajusta según veas el JSON real
        $tipoDocumento = $tipoDocRaw['codigo'] ?? ($tipoDocRaw['nombre'] ?? json_encode($tipoDocRaw, JSON_UNESCAPED_UNICODE));
    } elseif ($tipoDocRaw !== null) {
        $tipoDocumento = (string)$tipoDocRaw;
    }

    // Normalizar nombre de institución
    $nombreInst = null;
    if (is_array($nombreInstRaw)) {
        $nombreInst = $nombreInstRaw['nombre'] ?? json_encode($nombreInstRaw, JSON_UNESCAPED_UNICODE);
    } elseif ($nombreInstRaw !== null) {
        $nombreInst = (string)$nombreInstRaw;
    }

    // Porcentaje de exoneración
    $porcentaje = null;
    if (isset($data['porcentajeExoneracion'])) {
        $porcentaje = (float)$data['porcentajeExoneracion'];
    } elseif (isset($data['impuesto'])) {
        $porcentaje = (float)$data['impuesto'];
    }

    $fechaInicio = $data['fechaInicio'] ?? ($data['fechaVigenciaDesde'] ?? null);
    $fechaFin    = $data['fechaFin'] ?? ($data['fechaVigenciaHasta'] ?? null);

    $estadoRaw = $data['estado'] ?? ($data['situacion'] ?? null);
    $estado = is_array($estadoRaw)
        ? ($estadoRaw['codigo'] ?? ($estadoRaw['nombre'] ?? json_encode($estadoRaw, JSON_UNESCAPED_UNICODE)))
        : ($estadoRaw !== null ? (string)$estadoRaw : null);

    $condicionRaw = $data['condicion'] ?? null;
    $condicion = is_array($condicionRaw)
        ? ($condicionRaw['descripcion'] ?? json_encode($condicionRaw, JSON_UNESCAPED_UNICODE))
        : ($condicionRaw !== null ? (string)$condicionRaw : null);

    $cabys = null;
    if (isset($data['cabysAutorizados']) && is_array($data['cabysAutorizados'])) {
        $cabys = $data['cabysAutorizados'];
    } elseif (isset($data['detalleCabys']) && is_array($data['detalleCabys'])) {
        $cabys = $data['detalleCabys'];
    }

    return [
        'autorizacion'           => $autorizacion,
        'numero_documento'       => $numeroDocumento,
        'tipo_documento'         => $tipoDocumento,      // ✅ ahora siempre es string o null
        'nombre_institucion'     => $nombreInst,
        'porcentaje_exoneracion' => $porcentaje,
        'fecha_inicio'           => $fechaInicio,
        'fecha_fin'              => $fechaFin,
        'estado'                 => $estado,
        'condicion'              => $condicion,
        'cabys_autorizados'      => $cabys,
    ];
}


    /**
     * @param array<string,mixed>|array<int,mixed>|null $cabys
     * @param array<string,mixed>|array<int,mixed>|null $raw
     */
    private static function guardarResultado(
        string $autorizacion,
        ?string $numeroDocumento,
        ?string $tipoDocumento,
        ?string $nombreInstitucion,
        ?float $porcentaje,
        ?string $fechaInicio,
        ?string $fechaFin,
        ?array $cabys,
        $raw,
        ?int $httpCode,
        ?string $errorMessage,
        ?string $estado = null,
        ?string $condicion = null
    ): void {
        $pdo = Connection::getPdo();

        $sql = "
            INSERT INTO exoneraciones_fe
              (autorizacion, numero_documento, tipo_documento, nombre_institucion,
               porcentaje_exoneracion, fecha_inicio, fecha_fin, estado, condicion,
               cabys_autorizados, fuente, raw_response,
               ultimo_http_code, error_message, creado_en, ultimo_intento)
            VALUES
              (:autorizacion, :num_doc, :tipo_doc, :nombre_inst,
               :porcentaje, :fec_ini, :fec_fin, :estado, :condicion,
               :cabys, 'hacienda', :raw,
               :http_code, :error_msg, NOW(), NOW())
            ON DUPLICATE KEY UPDATE
              numero_documento       = VALUES(numero_documento),
              tipo_documento         = VALUES(tipo_documento),
              nombre_institucion     = VALUES(nombre_institucion),
              porcentaje_exoneracion = VALUES(porcentaje_exoneracion),
              fecha_inicio           = VALUES(fecha_inicio),
              fecha_fin              = VALUES(fecha_fin),
              estado                 = VALUES(estado),
              condicion              = VALUES(condicion),
              cabys_autorizados      = VALUES(cabys_autorizados),
              raw_response           = VALUES(raw_response),
              ultimo_http_code       = VALUES(ultimo_http_code),
              error_message          = VALUES(error_message),
              actualizado_en         = NOW(),
              ultimo_intento         = NOW()
        ";

        $stmt = $pdo->prepare($sql);

        $cabysJson = $cabys !== null ? json_encode($cabys, JSON_UNESCAPED_UNICODE) : null;
        $rawJson   = $raw !== null ? json_encode($raw, JSON_UNESCAPED_UNICODE) : null;

        $stmt->execute([
            ':autorizacion' => $autorizacion,
            ':num_doc'      => $numeroDocumento,
            ':tipo_doc'     => $tipoDocumento,
            ':nombre_inst'  => $nombreInstitucion,
            ':porcentaje'   => $porcentaje,
            ':fec_ini'      => $fechaInicio,
            ':fec_fin'      => $fechaFin,
            ':estado'       => $estado,
            ':condicion'    => $condicion,
            ':cabys'        => $cabysJson,
            ':raw'          => $rawJson,
            ':http_code'    => $httpCode,
            ':error_msg'    => $errorMessage,
        ]);
    }

    /**
     * Hace GET y devuelve JSON + HTTP code
     *
     * @param-out int $httpCode
     * @return array<string,mixed>|array<int,mixed>|null
     */
    private static function httpGetJson(string $url, ?int &$httpCode): ?array
    {
        $httpCode = 0;

        $ch = curl_init();
        if ($ch === false) {
            throw new RuntimeException('No se pudo inicializar cURL');
        }

        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT        => 10,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
        ]);

        $response = curl_exec($ch);

        if ($response === false) {
            $httpCode = 0;
            curl_close($ch);
            return null;
        }

        $httpCode = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $data = json_decode($response, true);
        if (!is_array($data)) {
            return null;
        }

        return $data;
    }
}
