<?php
declare(strict_types=1);

namespace App\Services;

use App\Database\Connection;
use RuntimeException;
use Stibenamm\FirmaXadesCR\Firmador;

class FirmaService
{
    /**
     * Firma un XML de comprobante electrónico usando XAdES-EPES
     * según los requisitos de Hacienda CR.
     *
     * @param string $xml        XML de la factura/ND/NC SIN firma.
     * @param int    $clienteId  ID del cliente (multi-tenant).
     *
     * @return string XML FIRMADO (listo para enviar a Hacienda).
     */
    public static function firmarXmlParaCliente(string $xml, int $clienteId): string
    {
        if (trim($xml) === '') {
            throw new RuntimeException('XML vacío, no se puede firmar.');
        }

        // 1) Obtener datos del certificado del cliente desde la BD
        //    *** USANDO TU TABLA REAL: clientes_api ***
        $pdo = Connection::getPdo();

        $sql = "
            SELECT 
                cert_path,
                cert_pin
            FROM clientes_api
            WHERE id = :cid
            LIMIT 1
        ";

        $stmt = $pdo->prepare($sql);
        $stmt->execute([':cid' => $clienteId]);
        $row = $stmt->fetch(\PDO::FETCH_ASSOC);

        if (!$row) {
            throw new RuntimeException(
                'No se encontró configuración de certificado para cliente_id=' . $clienteId
            );
        }

        $pfxPath = (string)($row['cert_path'] ?? '');
        $pin     = (string)($row['cert_pin'] ?? '');

        if ($pfxPath === '' || $pin === '') {
            throw new RuntimeException(
                'Configuración de certificado incompleta para cliente_id=' . $clienteId
                . ' (cert_path o cert_pin vacío).'
            );
        }

        // Si guardas ruta relativa en DB, aquí podrías resolverla contra la raíz del proyecto.
        // Por ahora asumimos que cert_path ya es una ruta absoluta válida en el servidor.
        if (!is_file($pfxPath)) {
            throw new RuntimeException(
                'El archivo de certificado (.p12/.pfx) no existe en la ruta: ' . $pfxPath
            );
        }

        // 2) Firmar usando firma-xades-cr
        try {
            $firmador = new Firmador();

            $xmlFirmado = $firmador->firmarXml(
                $pfxPath,
                $pin,
                $xml,
                $firmador::TO_XML_STRING
            );

            if (!is_string($xmlFirmado) || trim($xmlFirmado) === '') {
                throw new RuntimeException('El firmador devolvió un XML vacío.');
            }

            return $xmlFirmado;

        } catch (\Throwable $e) {
            throw new RuntimeException(
                'Error al firmar XML para cliente_id=' . $clienteId . ': ' . $e->getMessage(),
                (int)$e->getCode(),
                $e
            );
        }
    }
}
