<?php
declare(strict_types=1);

namespace App\Services;

use DOMDocument;
use DOMElement;
use DateTimeImmutable;
use DateTimeZone;
use RuntimeException;

class XmlFacturaBuilder
{
    /**
     * Construye el XML de la factura electrónica FE v4.4
     *
     * @param array  $data              JSON decodificado del request
     * @param string $clave             Clave generada
     * @param string $numeroConsecutivo Número consecutivo generado
     * @param string $ambiente          'pruebas'|'produccion'
     */
    public static function construirFacturaXml(
        array $data,
        string $clave,
        string $numeroConsecutivo,
        string $ambiente
    ): string {
        $doc = new DOMDocument('1.0', 'utf-8');
        $doc->preserveWhiteSpace = false;
        $doc->formatOutput       = false;

        $nsFe = 'https://cdn.comprobanteselectronicos.go.cr/xml-schemas/v4.4/facturaElectronica';

        /** @var DOMElement $root */
        $root = $doc->createElementNS($nsFe, 'FacturaElectronica');
        $root->setAttribute('xmlns:xsi', 'http://www.w3.org/2001/XMLSchema-instance');
        $root->setAttribute(
            'xsi:schemaLocation',
            'https://cdn.comprobanteselectronicos.go.cr/xml-schemas/v4.4/facturaElectronica ' .
            'https://cdn.comprobanteselectronicos.go.cr/xml-schemas/v4.4/facturaElectronica.xsd'
        );
        $doc->appendChild($root);

        // ========= ENCABEZADO =========
        self::agregarEncabezado($doc, $root, $data, $clave, $numeroConsecutivo, $ambiente);

        // ========= DETALLE SERVICIO =========
        $detalle      = $data['detalle'] ?? $data['lineas'] ?? [];
        $detalleArray = is_array($detalle) ? $detalle : [];
        self::agregarDetalleServicio($doc, $root, $detalleArray);

        // ========= RESUMEN FACTURA =========
        self::agregarResumenFactura($doc, $root, $data, $detalleArray);

        // ========= OTROS (opcional, uso comercial) =========
        self::agregarOtros($doc, $root, $data);

        return $doc->saveXML();
    }

    // -----------------------------------------------------------------
    // HELPERS BÁSICOS
    // -----------------------------------------------------------------

    private static function addChild(
        DOMDocument $doc,
        DOMElement $parent,
        string $name,
        ?string $value = null
    ): DOMElement {
        $el = $doc->createElementNS($parent->namespaceURI, $name);
        if ($value !== null && $value !== '') {
            $el->appendChild($doc->createTextNode($value));
        }
        $parent->appendChild($el);
        return $el;
    }

    private static function formatDecimal(float $value, int $decimals = 5): string
    {
        return number_format($value, $decimals, '.', '');
    }

    private static function nowCostaRica(): string
    {
        $tz  = new DateTimeZone('America/Costa_Rica');
        $now = new DateTimeImmutable('now', $tz);
        return $now->format('Y-m-d\TH:i:sP');
    }

    // -----------------------------------------------------------------
    // ENCABEZADO
    // -----------------------------------------------------------------

    private static function agregarEncabezado(
        DOMDocument $doc,
        DOMElement $root,
        array $data,
        string $clave,
        string $numeroConsecutivo,
        string $ambiente
    ): void {
        $emisorData   = $data['emisor']   ?? [];
        $receptorData = $data['receptor'] ?? [];

        $tipoIdEmisor = (string)(
            $emisorData['tipo_identificacion']
            ?? ($emisorData['identificacion']['tipo'] ?? '01')
        );

        $numIdEmisor = (string)(
            $emisorData['numero_identificacion']
            ?? ($emisorData['identificacion']['numero'] ?? '000000000')
        );

        // 1) Clave
        self::addChild($doc, $root, 'Clave', $clave);

        // 2) ProveedorSistemas (si lo usas comercialmente; Hacienda lo ignora a nivel tributario)
        $proveedorSistemas = (string)($data['proveedor_sistemas'] ?? $numIdEmisor);
        self::addChild($doc, $root, 'ProveedorSistemas', $proveedorSistemas);

        // 3) CodigoActividadEmisor (código de actividad económica)
        $codigoActividad = (string)($data['codigo_actividad_emisor'] ?? $data['codigo_actividad'] ?? '000000');
        self::addChild($doc, $root, 'CodigoActividadEmisor', $codigoActividad);

        // 4) NumeroConsecutivo
        self::addChild($doc, $root, 'NumeroConsecutivo', $numeroConsecutivo);

        // 5) FechaEmision
        $fechaEmision = $data['fecha_emision'] ?? self::nowCostaRica();
        self::addChild($doc, $root, 'FechaEmision', $fechaEmision);

        // ===== EMISOR =====
        $emisor = self::addChild($doc, $root, 'Emisor');

        $nombreEmisor = (string)($emisorData['nombre'] ?? 'DESCONOCIDO');
        self::addChild($doc, $emisor, 'Nombre', $nombreEmisor);

        $identificacion = self::addChild($doc, $emisor, 'Identificacion');
        self::addChild($doc, $identificacion, 'Tipo', $tipoIdEmisor);
        self::addChild($doc, $identificacion, 'Numero', $numIdEmisor);

        if (!empty($emisorData['nombre_comercial'])) {
            self::addChild($doc, $emisor, 'NombreComercial', (string)$emisorData['nombre_comercial']);
        }

        if (!empty($emisorData['ubicacion']) && is_array($emisorData['ubicacion'])) {
            $u = $emisorData['ubicacion'];
            $ubicacion = self::addChild($doc, $emisor, 'Ubicacion');

            $provincia = (string)($u['provincia'] ?? '1');
            self::addChild($doc, $ubicacion, 'Provincia', $provincia);

            $canton = str_pad((string)($u['canton'] ?? '1'), 2, '0', STR_PAD_LEFT);
            self::addChild($doc, $ubicacion, 'Canton', $canton);

            $distrito = str_pad((string)($u['distrito'] ?? '1'), 2, '0', STR_PAD_LEFT);
            self::addChild($doc, $ubicacion, 'Distrito', $distrito);

            if (isset($u['barrio']) && $u['barrio'] !== '') {
                $barrio = preg_replace('/\D+/', '', (string)$u['barrio']);
                if (strlen($barrio) !== 5) {
                    throw new RuntimeException("Barrio inválido en Emisor: debe tener 5 dígitos");
                }
                self::addChild($doc, $ubicacion, 'Barrio', $barrio);
            }

            self::addChild($doc, $ubicacion, 'OtrasSenas', (string)($u['otras_senas'] ?? 'SIN DETALLE'));
        }

        if (!empty($emisorData['telefono']) && is_array($emisorData['telefono'])) {
            $t = $emisorData['telefono'];
            $telefono = self::addChild($doc, $emisor, 'Telefono');
            self::addChild($doc, $telefono, 'CodigoPais', (string)($t['codigo_pais'] ?? '506'));
            self::addChild(
                $doc,
                $telefono,
                'NumTelefono',
                (string)($t['num_telefono'] ?? $t['numero'] ?? '00000000')
            );
        }

        if (!empty($emisorData['correo'])) {
            self::addChild($doc, $emisor, 'CorreoElectronico', (string)$emisorData['correo']);
        }

        // ===== RECEPTOR =====
        if (!empty($receptorData)) {
            $receptor = self::addChild($doc, $root, 'Receptor');

            $nombreReceptor = (string)($receptorData['nombre'] ?? 'CLIENTE');
            self::addChild($doc, $receptor, 'Nombre', $nombreReceptor);

            if (
                !empty($receptorData['tipo_identificacion']) ||
                !empty($receptorData['numero_identificacion']) ||
                !empty($receptorData['identificacion'])
            ) {
                $idRec = self::addChild($doc, $receptor, 'Identificacion');

                $tipoIdRec = (string)(
                    $receptorData['tipo_identificacion']
                    ?? ($receptorData['identificacion']['tipo'] ?? '01')
                );
                $numIdRec = (string)(
                    $receptorData['numero_identificacion']
                    ?? ($receptorData['identificacion']['numero'] ?? '')
                );

                self::addChild($doc, $idRec, 'Tipo', $tipoIdRec);
                self::addChild($doc, $idRec, 'Numero', $numIdRec);
            }

            if (!empty($receptorData['ubicacion']) && is_array($receptorData['ubicacion'])) {
                $u = $receptorData['ubicacion'];
                $ubicacion = self::addChild($doc, $receptor, 'Ubicacion');

                $provincia = (string)($u['provincia'] ?? '1');
                self::addChild($doc, $ubicacion, 'Provincia', $provincia);

                $canton = str_pad((string)($u['canton'] ?? '1'), 2, '0', STR_PAD_LEFT);
                self::addChild($doc, $ubicacion, 'Canton', $canton);

                $distrito = str_pad((string)($u['distrito'] ?? '1'), 2, '0', STR_PAD_LEFT);
                self::addChild($doc, $ubicacion, 'Distrito', $distrito);

                if (isset($u['barrio']) && $u['barrio'] !== '') {
                    $barrio = preg_replace('/\D+/', '', (string)$u['barrio']);
                    if (strlen($barrio) !== 5) {
                        throw new RuntimeException("Barrio inválido en Receptor: debe tener 5 dígitos");
                    }
                    self::addChild($doc, $ubicacion, 'Barrio', $barrio);
                }

                self::addChild($doc, $ubicacion, 'OtrasSenas', (string)($u['otras_senas'] ?? 'SIN DETALLE'));
            }

            if (!empty($receptorData['telefono']) && is_array($receptorData['telefono'])) {
                $t = $receptorData['telefono'];
                $telefono = self::addChild($doc, $receptor, 'Telefono');
                self::addChild($doc, $telefono, 'CodigoPais', (string)($t['codigo_pais'] ?? '506'));
                self::addChild(
                    $doc,
                    $telefono,
                    'NumTelefono',
                    (string)($t['num_telefono'] ?? $t['numero'] ?? '00000000')
                );
            }

            if (!empty($receptorData['correo'])) {
                self::addChild($doc, $receptor, 'CorreoElectronico', (string)$receptorData['correo']);
            }
        }

        // ===== CONDICIÓN DE VENTA =====
        $condicionVenta = (string)($data['condicion_venta'] ?? '01');
        self::addChild($doc, $root, 'CondicionVenta', $condicionVenta);

        if ($condicionVenta === '99' && !empty($data['condicion_venta_otros'])) {
            self::addChild(
                $doc,
                $root,
                'CondicionVentaOtros',
                substr((string)$data['condicion_venta_otros'], 0, 40)
            );
        }

        if ($condicionVenta !== '01') {
            $plazoCredito = (string)($data['plazo_credito'] ?? '0');
            self::addChild($doc, $root, 'PlazoCredito', $plazoCredito);
        }
    }

    // -----------------------------------------------------------------
    // DETALLE SERVICIO
    // -----------------------------------------------------------------

    private static function agregarDetalleServicio(
        DOMDocument $doc,
        DOMElement $root,
        array $lineas
    ): void {
        $detalleServicio = self::addChild($doc, $root, 'DetalleServicio');

        $numero = 1;
        foreach ($lineas as $linea) {
            if (!is_array($linea)) {
                continue;
            }
            self::agregarLineaDetalle($doc, $detalleServicio, $linea, $numero);
            $numero++;
        }
    }

    private static function agregarLineaDetalle(
        DOMDocument $doc,
        DOMElement $detalleServicio,
        array $linea,
        int $numeroLinea
    ): void {
        $ns = $detalleServicio->namespaceURI;

        $lineaDetalle = $doc->createElementNS($ns, 'LineaDetalle');
        $detalleServicio->appendChild($lineaDetalle);

        // NumeroLinea
        $el = $doc->createElementNS($ns, 'NumeroLinea', (string)$numeroLinea);
        $lineaDetalle->appendChild($el);

        // CodigoCABYS (OBLIGATORIO y 13 dígitos)
        $codigoCabys = preg_replace(
            '/\D+/',
            '',
            (string)($linea['codigo_cabys'] ?? $linea['cabys'] ?? $linea['codigoCABYS'] ?? '')
        );
        if (strlen($codigoCabys) !== 13) {
            throw new RuntimeException("CodigoCABYS inválido en la línea {$numeroLinea}");
        }
        $el = $doc->createElementNS($ns, 'CodigoCABYS', $codigoCabys);
        $lineaDetalle->appendChild($el);

        // CodigoComercial (opcional, hasta 5)
        if (!empty($linea['codigo_comercial'])) {
            $codigos = is_array($linea['codigo_comercial'])
                ? array_slice($linea['codigo_comercial'], 0, 5)
                : [$linea['codigo_comercial']];

            foreach ($codigos as $cc) {
                if (!is_array($cc)) {
                    $cc = [
                        'tipo'   => '01',
                        'codigo' => (string)$cc,
                    ];
                }

                $codigoComercial = $doc->createElementNS($ns, 'CodigoComercial');
                $tipo            = $doc->createElementNS($ns, 'Tipo', substr((string)($cc['tipo'] ?? '01'), 0, 2));
                $codigo          = $doc->createElementNS($ns, 'Codigo', substr((string)($cc['codigo'] ?? ''), 0, 20));

                $codigoComercial->appendChild($tipo);
                $codigoComercial->appendChild($codigo);
                $lineaDetalle->appendChild($codigoComercial);
            }
        }

        // Cantidad
        $cantidad = (float)($linea['cantidad'] ?? 1);
        $el = $doc->createElementNS($ns, 'Cantidad', self::formatDecimal($cantidad, 3));
        $lineaDetalle->appendChild($el);

        // UnidadMedida
        $unidadMedida = (string)($linea['unidad_medida'] ?? 'Sp');
        $el = $doc->createElementNS($ns, 'UnidadMedida', $unidadMedida);
        $lineaDetalle->appendChild($el);

        // TipoTransaccion (opcional)
        if (!empty($linea['tipo_transaccion'])) {
            $tipoTransaccion = substr((string)$linea['tipo_transaccion'], 0, 2);
            $el = $doc->createElementNS($ns, 'TipoTransaccion', $tipoTransaccion);
            $lineaDetalle->appendChild($el);
        }

        // UnidadMedidaComercial (opcional)
        if (!empty($linea['unidad_medida_comercial'])) {
            $el = $doc->createElementNS(
                $ns,
                'UnidadMedidaComercial',
                substr((string)$linea['unidad_medida_comercial'], 0, 20)
            );
            $lineaDetalle->appendChild($el);
        }

        // Detalle
        $detalleTxt = trim((string)($linea['detalle'] ?? $linea['descripcion'] ?? 'Servicio'));
        if (strlen($detalleTxt) < 3) {
            $detalleTxt = str_pad($detalleTxt, 3, '.');
        }
        $el = $doc->createElementNS($ns, 'Detalle', substr($detalleTxt, 0, 200));
        $lineaDetalle->appendChild($el);

        // PrecioUnitario, MontoTotal
        $precioUnitario = (float)($linea['precio_unitario'] ?? $linea['precio'] ?? 0.0);
        $montoTotal     = $cantidad * $precioUnitario;

        $el = $doc->createElementNS($ns, 'PrecioUnitario', self::formatDecimal($precioUnitario, 5));
        $lineaDetalle->appendChild($el);

        $el = $doc->createElementNS($ns, 'MontoTotal', self::formatDecimal($montoTotal, 5));
        $lineaDetalle->appendChild($el);

        // Descuento (opcional)
        $montoDescuento = (float)(
            $linea['monto_descuento'] ?? $linea['descuento'] ?? $linea['descuento_monto'] ?? 0.0
        );
        if ($montoDescuento < 0) {
            $montoDescuento = 0.0;
        }

        if ($montoDescuento > 0) {
            $descuento = $doc->createElementNS($ns, 'Descuento');

            $montoDesc = $doc->createElementNS($ns, 'MontoDescuento', self::formatDecimal($montoDescuento, 5));
            $descuento->appendChild($montoDesc);

            $codNat = substr((string)($linea['codigo_descuento'] ?? '01'), 0, 2);
            $naturaleza = $doc->createElementNS($ns, 'CodigoNaturalezaDescuento', $codNat);
            $descuento->appendChild($naturaleza);

            $lineaDetalle->appendChild($descuento);
        }

        // SubTotal
        $subTotal = $montoTotal - $montoDescuento;
        $el = $doc->createElementNS($ns, 'SubTotal', self::formatDecimal($subTotal, 5));
        $lineaDetalle->appendChild($el);

        // IVACobradoFabrica (opcional)
        if (!empty($linea['iva_cobrado_fabrica'])) {
            $el = $doc->createElementNS($ns, 'IVACobradoFabrica', (string)$linea['iva_cobrado_fabrica']);
            $lineaDetalle->appendChild($el);
        }

        // BaseImponible
        $baseImponible = $subTotal;
        $baseEl = $doc->createElementNS($ns, 'BaseImponible', self::formatDecimal($baseImponible, 5));
        $lineaDetalle->appendChild($baseEl);

        // ============================================================
        // FIX HACIENDA: Impuesto/ImpuestoAsumido deben ir ANTES de MontoTotalLinea
        // ============================================================

        // Normalizar impuesto array
        $impArr = is_array($linea['impuesto'] ?? null) ? $linea['impuesto'] : [];

        $codigoImpuesto = (string)(
            $impArr['codigo']
            ?? $linea['impuesto_codigo']
            ?? '01'
        );

        $tarifa = (float)(
            $impArr['tarifa']
            ?? $linea['impuesto_tarifa']
            ?? 0.0
        );

        $montoImpuesto = $tarifa > 0
            ? round($baseImponible * ($tarifa / 100), 5)
            : 0.0;

        $codigoTarifa = (string)(
            $impArr['codigo_tarifa_iva']
            ?? $linea['impuesto_codigo_tarifa']
            ?? ''
        );

        if ($codigoTarifa === '') {
            if ($tarifa > 0.0 && $codigoImpuesto === '01') {
                $codigoTarifa = '08'; // 13%
            } elseif ($tarifa == 0.0) {
                $codigoTarifa = '07'; // 0%
            } else {
                $codigoTarifa = '01';
            }
        }

        // ImpuestoAsumidoEmisorFabrica (opcional)
        $impAsumido = (float)($linea['impuesto_asumido_emisor'] ?? 0.0);
        if ($impAsumido < 0) {
            $impAsumido = 0.0;
        }

        // Si por alguna razón ya existía MontoTotalLinea (por builds previos), eliminarlo para reordenar
        $existingMontoTotalLinea = null;
        foreach ($lineaDetalle->childNodes as $child) {
            if ($child instanceof DOMElement && $child->localName === 'MontoTotalLinea') {
                $existingMontoTotalLinea = $child;
                break;
            }
        }
        if ($existingMontoTotalLinea) {
            $lineaDetalle->removeChild($existingMontoTotalLinea);
        }

        // Crear SIEMPRE Impuesto (aunque sea 0) y agregarlo antes de MontoTotalLinea
        $impuestoEl = $doc->createElementNS($ns, 'Impuesto');
        $impuestoEl->appendChild($doc->createElementNS($ns, 'Codigo', substr($codigoImpuesto, 0, 2)));
        $impuestoEl->appendChild($doc->createElementNS($ns, 'CodigoTarifaIVA', substr($codigoTarifa, 0, 2)));
        $impuestoEl->appendChild($doc->createElementNS($ns, 'Tarifa', self::formatDecimal($tarifa, 2)));
        $impuestoEl->appendChild($doc->createElementNS($ns, 'Monto', self::formatDecimal($montoImpuesto, 5)));
        $lineaDetalle->appendChild($impuestoEl);

        // ImpuestoAsumidoEmisorFabrica (opcional) también va antes de MontoTotalLinea
        if ($impAsumido > 0) {
            $asumidoEl = $doc->createElementNS(
                $ns,
                'ImpuestoAsumidoEmisorFabrica',
                self::formatDecimal($impAsumido, 5)
            );
            $lineaDetalle->appendChild($asumidoEl);
        }

        // MontoTotalLinea al final: SubTotal + (Impuesto - ImpuestoAsumidoEmisorFabrica)
        $montoTotalLinea = $subTotal + ($montoImpuesto - $impAsumido);
        $montoTotalLineaEl = $doc->createElementNS($ns, 'MontoTotalLinea', self::formatDecimal($montoTotalLinea, 5));
        $lineaDetalle->appendChild($montoTotalLineaEl);
    }

    // -----------------------------------------------------------------
    // CÁLCULO DE TOTALES DESDE DETALLE
    // -----------------------------------------------------------------

    private static function calcularTotalesDesdeDetalle(array $detalle): array
    {
        $totalServGravados      = 0.0;
        $totalServExentos       = 0.0;
        $totalServNoSujeto      = 0.0;

        $totalMercGravadas      = 0.0;
        $totalMercExentas       = 0.0;
        $totalMercNoSujetas     = 0.0;

        $totalDescuentos        = 0.0;
        $totalImpuesto          = 0.0;

        foreach ($detalle as $linea) {
            if (!is_array($linea)) {
                continue;
            }

            $cantidad       = (float)($linea['cantidad'] ?? 1);
            $precioUnitario = (float)($linea['precio_unitario'] ?? $linea['precio'] ?? 0.0);
            $montoTotal     = $cantidad * $precioUnitario;

            $montoDescuento = (float)(
                $linea['monto_descuento'] ?? $linea['descuento'] ?? $linea['descuento_monto'] ?? 0.0
            );
            if ($montoDescuento < 0) {
                $montoDescuento = 0.0;
            }

            $subTotal = $montoTotal - $montoDescuento;

            $impArr = is_array($linea['impuesto'] ?? null) ? $linea['impuesto'] : [];
            $tarifa = (float)(
                $impArr['tarifa']
                ?? $linea['impuesto_tarifa']
                ?? 0.0
            );

            $impuestoLinea = $tarifa > 0
                ? round($subTotal * ($tarifa / 100), 5)
                : 0.0;

            // Para simplificar: todo lo tratamos como servicios
            if ($tarifa > 0.0) {
                $totalServGravados += $subTotal;
            } else {
                $totalServExentos += $subTotal;
            }

            $totalDescuentos += $montoDescuento;
            $totalImpuesto   += $impuestoLinea;
        }

        $totalGravado   = $totalServGravados + $totalMercGravadas;
        $totalExento    = $totalServExentos  + $totalMercExentas;
        $totalNoSujeto  = $totalServNoSujeto + $totalMercNoSujetas;

        $totalVenta       = $totalGravado + $totalExento + $totalNoSujeto;
        $totalVentaNeta   = $totalVenta - $totalDescuentos;
        $totalComprobante = $totalVentaNeta + $totalImpuesto;

        return [
            'TotalServGravados'        => $totalServGravados,
            'TotalServExentos'         => $totalServExentos,
            'TotalServNoSujeto'        => $totalServNoSujeto,

            'TotalMercanciasGravadas'  => $totalMercGravadas,
            'TotalMercanciasExentas'   => $totalMercExentas,
            'TotalMercNoSujeta'        => $totalMercNoSujetas,

            'TotalGravado'             => $totalGravado,
            'TotalExento'              => $totalExento,
            'TotalExonerado'           => 0.0,
            'TotalNoSujeto'            => $totalNoSujeto,

            'TotalVenta'               => $totalVenta,
            'TotalDescuentos'          => $totalDescuentos,
            'TotalVentaNeta'           => $totalVentaNeta,
            'TotalImpuesto'            => $totalImpuesto,
            'TotalComprobante'         => $totalComprobante,
        ];
    }

    // -----------------------------------------------------------------
    // RESUMEN FACTURA
    // -----------------------------------------------------------------

    private static function agregarResumenFactura(
        DOMDocument $doc,
        DOMElement $root,
        array $data,
        array $detalle
    ): void {
        $ns = $root->namespaceURI;

        $resumen = $doc->createElementNS($ns, 'ResumenFactura');
        $root->appendChild($resumen);

        // Moneda
        $codigoMoneda = (string)($data['moneda'] ?? 'CRC');
        $tipoCambio   = (float)($data['tipo_cambio'] ?? 1);

        $moneda = $doc->createElementNS($ns, 'CodigoTipoMoneda');
        $resumen->appendChild($moneda);

        $el = $doc->createElementNS($ns, 'CodigoMoneda', $codigoMoneda);
        $moneda->appendChild($el);

        $el = $doc->createElementNS($ns, 'TipoCambio', self::formatDecimal($tipoCambio, 5));
        $moneda->appendChild($el);

        if (!empty($detalle)) {
            $totales = self::calcularTotalesDesdeDetalle($detalle);

            $totalServGravados  = $totales['TotalServGravados'];
            $totalServExentos   = $totales['TotalServExentos'];
            $totalServNoSujeto  = $totales['TotalServNoSujeto'];

            $totalMercGravadas  = $totales['TotalMercanciasGravadas'];
            $totalMercExentas   = $totales['TotalMercanciasExentas'];
            $totalMercNoSujetas = $totales['TotalMercNoSujeta'];

            $totalGravado       = $totales['TotalGravado'];
            $totalExento        = $totales['TotalExento'];
            $totalExonerado     = $totales['TotalExonerado'];
            $totalNoSujeto      = $totales['TotalNoSujeto'];

            $totalVenta         = $totales['TotalVenta'];
            $totalDescuentos    = $totales['TotalDescuentos'];
            $totalVentaNeta     = $totales['TotalVentaNeta'];
            $totalImpuesto      = $totales['TotalImpuesto'];
            $totalComprobante   = $totales['TotalComprobante'];
        } else {
            $totalServGravados  = (float)($data['total_serv_gravados'] ?? 0.0);
            $totalServExentos   = (float)($data['total_serv_exentos'] ?? 0.0);
            $totalServNoSujeto  = (float)($data['total_serv_no_sujeto'] ?? 0.0);

            $totalMercGravadas  = (float)($data['total_merc_gravadas'] ?? 0.0);
            $totalMercExentas   = (float)($data['total_merc_exentas'] ?? 0.0);
            $totalMercNoSujetas = (float)($data['total_merc_no_sujetas'] ?? 0.0);

            $totalGravado       = (float)($data['total_gravado'] ?? 0.0);
            $totalExento        = (float)($data['total_exento'] ?? 0.0);
            $totalExonerado     = (float)($data['total_exonerado'] ?? 0.0);
            $totalNoSujeto      = (float)($data['total_no_sujeto'] ?? 0.0);

            $totalVenta         = (float)($data['total_venta'] ?? 0.0);
            $totalDescuentos    = (float)($data['total_descuentos'] ?? 0.0);
            $totalVentaNeta     = (float)($data['total_venta_neta'] ?? 0.0);
            $totalImpuesto      = (float)($data['total_impuesto'] ?? 0.0);
            $totalComprobante   = (float)($data['total_comprobante'] ?? 0.0);

            if ($totalVenta <= 0.0 && $totalGravado + $totalExento + $totalNoSujeto > 0.0) {
                $totalVenta       = $totalGravado + $totalExento + $totalNoSujeto;
                $totalVentaNeta   = $totalVenta - $totalDescuentos;
                $totalComprobante = $totalVentaNeta + $totalImpuesto;
            }
        }

        // Orden de campos alineado con FE v4.4
        $map = [
            'TotalServGravados'        => $totalServGravados,
            'TotalServExentos'         => $totalServExentos,
            'TotalServExonerado'       => 0.0,
            'TotalServNoSujeto'        => $totalServNoSujeto,

            'TotalMercanciasGravadas'  => $totalMercGravadas,
            'TotalMercanciasExentas'   => $totalMercExentas,
            'TotalMercExonerada'       => 0.0,
            'TotalMercNoSujeta'        => $totalMercNoSujetas,

            'TotalGravado'             => $totalGravado,
            'TotalExento'              => $totalExento,
            'TotalExonerado'           => $totalExonerado,
            'TotalNoSujeto'            => $totalNoSujeto,

            'TotalVenta'               => $totalVenta,
            'TotalDescuentos'          => $totalDescuentos,
            'TotalVentaNeta'           => $totalVentaNeta,
            'TotalImpuesto'            => $totalImpuesto,
        ];

        foreach ($map as $tag => $valor) {
            $el = $doc->createElementNS($ns, $tag, self::formatDecimal((float)$valor, 5));
            $resumen->appendChild($el);
        }

        // TotalDesgloseImpuestos (nuevo en 4.4, Hacienda lo pide)
        $el = $doc->createElementNS($ns, 'TotalDesgloseImpuestos', self::formatDecimal($totalImpuesto, 5));
        $resumen->appendChild($el);

        // ===== MedioPago v4.4 =====
        $condicionVenta = (string)($data['condicion_venta'] ?? '01');

        if (!in_array($condicionVenta, ['02', '08', '10'], true)) {
            $mediosPagoDetallados = $data['medios_pago_detalle'] ?? null;

            if (is_array($mediosPagoDetallados) && !empty($mediosPagoDetallados)) {
                foreach ($mediosPagoDetallados as $mp) {
                    if (!is_array($mp)) {
                        continue;
                    }
                    $tipo  = (string)($mp['tipo'] ?? $mp['codigo'] ?? '01');
                    $monto = isset($mp['monto']) ? (float)$mp['monto'] : 0.0;

                    $medioPagoEl = $doc->createElementNS($ns, 'MedioPago');
                    $resumen->appendChild($medioPagoEl);

                    $tipoEl = $doc->createElementNS($ns, 'TipoMedioPago', substr($tipo, 0, 2));
                    $medioPagoEl->appendChild($tipoEl);

                    if ($tipo === '99' && !empty($mp['detalle'])) {
                        $otrosEl = $doc->createElementNS(
                            $ns,
                            'MedioPagoOtros',
                            substr((string)$mp['detalle'], 0, 100)
                        );
                        $medioPagoEl->appendChild($otrosEl);
                    }

                    if ($monto <= 0.0) {
                        $monto = $totalComprobante;
                    }

                    $montoEl = $doc->createElementNS(
                        $ns,
                        'TotalMedioPago',
                        self::formatDecimal($monto, 5)
                    );
                    $medioPagoEl->appendChild($montoEl);
                }
            } else {
                $mediosPago = $data['medio_pago'] ?? $data['medios_pago'] ?? ['01'];
                if (!is_array($mediosPago)) {
                    $mediosPago = [$mediosPago];
                }
                $mediosPago = array_slice($mediosPago, 0, 4);

                $numMedios  = max(count($mediosPago), 1);
                $montoBase  = $totalComprobante > 0 ? $totalComprobante : 1.0;

                foreach ($mediosPago as $mp) {
                    $tipo = is_array($mp)
                        ? (string)($mp['tipo'] ?? $mp['codigo'] ?? '01')
                        : (string)$mp;

                    $medioPagoEl = $doc->createElementNS($ns, 'MedioPago');
                    $resumen->appendChild($medioPagoEl);

                    $tipoEl = $doc->createElementNS($ns, 'TipoMedioPago', substr($tipo, 0, 2));
                    $medioPagoEl->appendChild($tipoEl);

                    if (is_array($mp) && $tipo === '99' && !empty($mp['detalle'])) {
                        $otrosEl = $doc->createElementNS(
                            $ns,
                            'MedioPagoOtros',
                            substr((string)$mp['detalle'], 0, 100)
                        );
                        $medioPagoEl->appendChild($otrosEl);
                    }

                    $montoEl = $doc->createElementNS(
                        $ns,
                        'TotalMedioPago',
                        self::formatDecimal($montoBase, 5)
                    );
                    $medioPagoEl->appendChild($montoEl);
                }
            }
        }

        // TotalComprobante al final
        $el = $doc->createElementNS($ns, 'TotalComprobante', self::formatDecimal($totalComprobante, 5));
        $resumen->appendChild($el);
    }

    // -----------------------------------------------------------------
    // OTROS (uso comercial, no tributario)
    // -----------------------------------------------------------------

    private static function agregarOtros(
        DOMDocument $doc,
        DOMElement $root,
        array $data
    ): void {
        $ns = $root->namespaceURI;

        if (empty($data['otros']) || !is_array($data['otros'])) {
            return;
        }

        $otros = $doc->createElementNS($ns, 'Otros');
        $root->appendChild($otros);

        foreach ($data['otros'] as $otro) {
            if (!is_array($otro)) {
                continue;
            }
            $otroEl = $doc->createElementNS($ns, 'OtroTexto');
            if (!empty($otro['codigo'])) {
                $otroEl->setAttribute('codigo', (string)$otro['codigo']);
            }
            $valor = (string)($otro['valor'] ?? '');
            if ($valor !== '') {
                $otroEl->appendChild($doc->createTextNode($valor));
            }
            $otros->appendChild($otroEl);
        }
    }
}
