<?php
declare(strict_types=1);

// Ruta base del proyecto (carpeta /public)
define('BASE_PATH', __DIR__);

// 1) Autoload de Composer (xmlseclibs, etc.)
require BASE_PATH . '/vendor/autoload.php';

// 2) Bootstrap propio del proyecto (config, router, etc.)
require BASE_PATH . '/src/bootstrap.php';

use App\Core\Router;
use App\Http\Response;
use App\Http\HttpException;

// Siempre respondemos JSON
header('Content-Type: application/json; charset=utf-8');

try {
    $router = new Router();

    // ---------- RUTAS ----------

    // Ping simple
    $router->add('GET', '/ping', function () {
        Response::json([
            'ok'      => true,
            'message' => 'Framework AFE corriendo correctamente 🚀',
        ]);
    });

    // Crear documento (JSON → BD + clave + XML + firma)
    $router->add(
        'POST',
        '/api/v1/documentos',
        [\App\Controllers\DocumentController::class, 'store']
    );

    // Enviar documento firmado a Hacienda
    $router->add(
        'POST',
        '/api/v1/documentos/enviar',
        [\App\Controllers\DocumentController::class, 'send']
    );

    // Consultar estado de un documento
    $router->add(
        'GET',
        '/api/v1/documentos/estado',
        [\App\Controllers\DocumentController::class, 'status']
    );

    // Ver cabecera + detalle de un documento
    $router->add(
        'GET',
        '/api/v1/documentos/ver',
        [\App\Controllers\DocumentController::class, 'show']
    );

    // CABYS: búsqueda por término
    $router->add(
        'GET',
        '/api/v1/cabys/buscar',
        [\App\Controllers\CabysController::class, 'search']
    );

    // CABYS: obtener por código (y cachear si no existe)
    $router->add(
        'GET',
        '/api/v1/cabys/get',
        [\App\Controllers\CabysController::class, 'get']
    );
    
    // Contribuyentes (AE): consulta por identificación
    $router->add(
        'GET',
        '/api/v1/contribuyentes/consultar',
        [\App\Controllers\ContribuyenteController::class, 'consultar']
    );

    // Exoneraciones (fe/ex): consulta por autorización
    $router->add(
        'GET',
        '/api/v1/exoneraciones/consultar',
        [\App\Controllers\ExoneracionController::class, 'consultar']
    );

    // Tipo de cambio (indicadores/tc): USD y EUR
    $router->add(
        'GET',
        '/api/v1/indicadores/tc',
        [\App\Controllers\TipoCambioController::class, 'actual']
    );

    // Probar credenciales de Hacienda (sandbox)
    $router->add(
        'POST',
        '/api/v1/mh/test-credenciales',
        [\App\Controllers\MhController::class, 'testCredenciales']
    );

    // 🔚 Despachar la ruta actual (siempre al final)
    $router->dispatch();

} catch (HttpException $e) {
    // Errores de negocio / validación controlados
    Response::json([
        'ok'    => false,
        'error' => $e->getMessage(),
    ], $e->getStatusCode());
} catch (\Throwable $e) {
    // Errores fatales / inesperados
    http_response_code(500);
    echo json_encode([
        'ok'    => false,
        'error' => $e->getMessage(),
        'file'  => $e->getFile(),
        'line'  => $e->getLine(),
    ], JSON_UNESCAPED_UNICODE);
    exit;
}
