<?php

namespace App\Services;

use Illuminate\Support\Facades\DB;
use App\HttpClients\Factories\TipoCambioClientFactory;
use DateTimeImmutable;
use DateTimeZone;

class TipoCambioService
{
    /**
     * @param array $providers orden de fallback, ej: ['hacienda','bccr']
     * @param string $moneda 'CRC'|'USD'|'EUR'
     * @return array
     */
    public function obtenerCambioActual(array $providers, string $moneda = 'CRC'): array
    {
        $moneda = strtoupper(trim($moneda));
        if (!in_array($moneda, ['CRC', 'USD', 'EUR'], true)) {
            throw new \InvalidArgumentException("Moneda inválida: {$moneda}");
        }

        $date = new DateTimeImmutable('today', new DateTimeZone('America/Costa_Rica'));
        $fecha = $date->format('Y-m-d');

        // CRC base siempre 1
        if ($moneda === 'CRC') {
            // opcional: persistir fila CRC para consistencia histórica
            DB::table('tipo_cambio')->updateOrInsert(
                ['fecha' => $fecha, 'moneda' => 'CRC'],
                ['compra' => 1.0, 'venta' => 1.0, 'fuente' => 'base', 'raw_response' => null]
            );

            return $this->formatResponse('CRC', $fecha, 1.0, 1.0);
        }

        // Cache por fecha+moneda
        $cached = DB::table('tipo_cambio')
            ->where('fecha', $fecha)
            ->where('moneda', $moneda)
            ->first();

        if ($cached && $cached->venta !== null) {
            return $this->formatResponse(
                $moneda,
                $cached->fecha,
                (float) $cached->venta,
                (float) $cached->compra
            );
        }

        foreach ($providers as $provider) {
            $client = TipoCambioClientFactory::make($provider);
            $dto = $client->getActual($moneda);

            // si el proveedor no devuelve valor, intenta siguiente
            if ($dto->venta === null && $dto->compra === null) {
                continue;
            }

            DB::table('tipo_cambio')->updateOrInsert(
                ['fecha' => $dto->fecha->format('Y-m-d'), 'moneda' => $dto->moneda],
                [
                    'venta'        => $dto->venta,
                    'compra'       => $dto->compra,
                    'fuente'       => $dto->fuente ?? $provider,
                    'raw_response' => $dto->rawResponse,
                ]
            );

            return $this->formatResponse(
                $dto->moneda,
                $dto->fecha->format('Y-m-d'),
                (float) $dto->venta,
                (float) $dto->compra
            );
        }

        throw new \RuntimeException("No tipo de cambio disponible para {$moneda}");
    }

    private function formatResponse(string $moneda, string $fecha, float $venta, float $compra): array
    {
        return [
            'base'   => 'CRC',
            'moneda' => $moneda,
            'fecha'  => $fecha,
            'venta'  => round($venta, 4),
            'compra' => round($compra, 4),
        ];
    }
}
