<?php

namespace App\Utils;

use App\Enums\TipoDocumento;
use InvalidArgumentException;
use DateTimeInterface;
use App\Enums\TipoIdentificacion;

class ConsecutiveGenerator
{
    public static function mapTipoDocumento(string $tipo): string
    {
        $tipoDoc = TipoDocumento::tryFrom($tipo);

        if ($tipoDoc) {
            return $tipoDoc->value;
        }
        
        throw new InvalidArgumentException("Tipo de documento '{$tipo}' no es válido.");
    }

    public static function generarNumeroConsecutivo(
        int $sucursal,
        int $terminal,
        string $tipoDocumento,
        int $secuencia
    ): string
    {
        $codigoTipo = self::mapTipoDocumento($tipoDocumento);

        $sucursalStr  = str_pad((string) $sucursal, 3, '0', STR_PAD_LEFT);
        $terminalStr  = str_pad((string) $terminal, 5, '0', STR_PAD_LEFT);
        $secuenciaStr = str_pad((string) $secuencia, 10, '0', STR_PAD_LEFT);

        return "{$sucursalStr}{$terminalStr}{$codigoTipo}{$secuenciaStr}";
    }

    public static function generarClave(
        DateTimeInterface $fecha,
        string $consecutivo,
        string $tipoId,
        string $identificacion,
        string $situacion = "1",
        string $codigoPais = "506",
    ): string
    {
        if (!ctype_digit($tipoId) || !TipoIdentificacion::tryFrom($tipoId)) 
        {
            throw new InvalidArgumentException(message: 'El tipo de identificación no es reconocido en la clave numérica');
        }

        $identificacion = str_pad($identificacion, 12, '0', STR_PAD_LEFT);

        $dia  = $fecha->format('d');
        $mes  = $fecha->format('m');
        $anio = $fecha->format('y');

        $codSecRandom = random_int(10000000, 99999999);

        return "{$codigoPais}{$dia}{$mes}{$anio}{$identificacion}{$consecutivo}{$situacion}{$codSecRandom}";
    }
}

